package org.msh.tb.export_rest.dbreader;

import org.jboss.seam.Component;
import org.msh.reports.ReportConfiguration;
import org.msh.reports.datatable.Row;
import org.msh.reports.query.DataTableQuery;
import org.msh.tb.export_rest.core.ExcelExportException;
import org.msh.tb.reports2.IndicatorReportFactory;

import javax.persistence.EntityManager;
import javax.persistence.Query;
import java.util.*;

/**
 * Created by rmemoria on 19/7/17.
 */
public class ReaderSQLQuery {

    private String fromTable;
    private String fromAlias;
    private String orderBy;
    private String keyField;
    private List<TableJoin> joins = new ArrayList<TableJoin>();
    private List<String> restrictions = new ArrayList<String>();
    private List<SQLField> fields = new ArrayList<SQLField>();
    private Map<String, Object> parameters = new HashMap<String, Object>();


    public ReaderSQLQuery addField(String field, String title) {
        addField(field, title, null);
        return this;
    }

    public ReaderSQLQuery addField(String field, String title, Class<? extends Enum> enumClass) {
        fields.add(new SQLField(field, title, enumClass));
        return this;
    }

    public ReaderSQLQuery addJoin(String table, String alias, String join) {
        joins.add(new TableJoin(table, alias, join, false));
        return this;
    }

    public ReaderSQLQuery addLeftJoin(String table, String alias, String join) {
        joins.add(new TableJoin(table, alias, join, true));
        return this;
    }

    /**
     * Calculate the number of records to be returned from the reader
     *
     * @return
     */
    public long calcRecordCount() {
        StringBuilder sql = new StringBuilder();

        sql.append("select count(*) from " + fromTable + " " + fromAlias + '\n');

        for (TableJoin join : joins) {
            if (!join.isLeftJoin()) {
                sql.append("join " + join.getTable() + " " + join.getAlias() + " on " + join.getJoin() + "\n");
            }
        }

        addSqlWhereClause(sql);

        EntityManager em = getEntityManager();

        Query qry = em.createNativeQuery(sql.toString());

        for (Map.Entry<String, Object> p : parameters.entrySet()) {
            qry.setParameter(p.getKey(), p.getValue());
        }

        Number count = (Number) qry.getSingleResult();

        return count.longValue();
    }


    /**
     * Calculate the maximum number of records per key. Useful when grouping by keys, when it is
     * necessary to know in advance the number of columns
     *
     * @return maximum number of grouped records per key
     */
    public int calcMaxGroupCount() {
        StringBuilder sql = new StringBuilder();
        sql.append("select count(*) from " + fromTable + " " + fromAlias + '\n');

        sql.append(generateSQLJoin());

        addSqlWhereClause(sql);

        sql.append("\ngroup by " + keyField + "\norder by count(*) desc");

        EntityManager em = getEntityManager();

        Query qry = em.createNativeQuery(sql.toString());

        qry.setMaxResults(1);

        for (Map.Entry<String, Object> p : parameters.entrySet()) {
            qry.setParameter(p.getKey(), p.getValue());
        }

        Number count = (Number) qry.getSingleResult();

        return count.intValue();
    }


    /**
     * Create a list of grouped names from the given field name. Usually used when calculating
     * the columns of a reader
     *
     * @param fieldName the field name whom value will return
     * @return list of value names
     */
    public List<String> calcGroupedValues(String fieldName) {
        StringBuilder sql = new StringBuilder();

        sql.append("select " + fieldName +
                " from " + fromTable + ' ' + fromAlias + '\n' +
                generateSQLJoin());

        addSqlWhereClause(sql);

        sql.append("\ngroup by " + fieldName + "\norder by " + fieldName);

        EntityManager em = getEntityManager();

        Query qry = em.createNativeQuery(sql.toString());

        for (Map.Entry<String, Object> p : parameters.entrySet()) {
            qry.setParameter(p.getKey(), p.getValue());
        }

        List<String> titles = qry.getResultList();

        return titles;
    }

    private String generateSQLJoin() {
        StringBuilder s = new StringBuilder();

        for (TableJoin join : joins) {
            if (!join.isLeftJoin()) {
                s.append("join " + join.getTable() + " " + join.getAlias() + " on " + join.getJoin())
                        .append('\n');
            }
        }

        return s.toString();
    }

    public List<Object[]> loadRecord(int iniResult, int maxResults) {
        // use a pure database connection
        // when using entityManager.createNativeQuery, there was conflicts with duplicated column names
        ReportConfiguration.instance().registerResourceProvider(IndicatorReportFactory.instance());
        org.msh.reports.query.SQLQuery qry = new org.msh.reports.query.SQLQuery();
        qry.setIniResult(iniResult);
        qry.setMaxResults(maxResults);

        for (Map.Entry<String, Object> p : getParameters().entrySet()) {
            qry.setParameter(p.getKey(), p.getValue());
        }

        DataTableQuery dt = qry.execute(createSQL());

        List<Object[]> res = new ArrayList<Object[]>(dt.getRowCount());
        for (Row row : dt.getRows()) {
            res.add(row.getValues().toArray());
        }

        return res;
    }

    /**
     * Create the SQL declaration to return the records
     *
     * @return
     */
    private String createSQL() {
        if (keyField == null) {
            throw new ExcelExportException("No key field specified");
        }

        StringBuilder s = new StringBuilder();

        s.append("select ").append(keyField);

        int index = 0;
        for (SQLField field : fields) {
            s.append(",").append(field.getField()).append(" f").append(index++);
        }

        s.append('\n')
                .append("from ")
                .append(fromTable)
                .append(' ')
                .append(fromAlias)
                .append('\n');

        for (TableJoin join : joins) {
            s.append(join.isLeftJoin() ? "left join " : "join ")
                    .append(join.getTable())
                    .append(' ')
                    .append(join.getAlias())
                    .append(" on ")
                    .append(join.getJoin())
                    .append('\n');
        }

        addSqlWhereClause(s);

        if (orderBy != null) {
            s.append(" order by ").append(orderBy);
        }

        return s.toString();
    }


    /**
     * Add the SQL where clause to the SQL builder
     *
     * @param builder
     */
    private void addSqlWhereClause(StringBuilder builder) {
        if (restrictions.size() > 0) {
            String delim = "\n where ";
            for (String restr : restrictions) {
                builder.append(delim).append(restr);
                delim = "\n and ";
            }
        }
    }


    /**
     * Add restriction that will be included in the WHERE clause
     *
     * @param restriction
     */
    public ReaderSQLQuery addRestriction(String restriction) {
        restrictions.add(restriction);
        return this;
    }

    protected EntityManager getEntityManager() {
        return (EntityManager) Component.getInstance("entityManager");
    }

    public boolean isTableAliasInUse(String alias) {
        if (alias.equals(fromAlias)) {
            return true;
        }

        TableJoin join = findJoinByAlias(alias);
        return join != null;
    }

    public TableJoin findJoinByAlias(String alias) {
        for (TableJoin join : joins) {
            if (join.getAlias().equals(alias)) {
                return join;
            }
        }
        return null;
    }

    public String getOrderBy() {
        return orderBy;
    }

    public ReaderSQLQuery orderBy(String orderBy) {
        this.orderBy = orderBy;
        return this;
    }

    public String getKeyField() {
        return keyField;
    }

    public void setKeyField(String keyField) {
        this.keyField = keyField;
    }

    public void addParameter(String name, Object value) {
        if (value instanceof Enum) {
            parameters.put(name, ((Enum) value).ordinal());
        } else {
            parameters.put(name, value);
        }
    }

    public Map<String, Object> getParameters() {
        return parameters;
    }

    public void setParameters(Map<String, Object> parameters) {
        this.parameters = parameters;
    }

    public ReaderSQLQuery from(String from, String alias) {
        this.fromTable = from;
        this.fromAlias = alias;
        return this;
    }

    public String getFromTable() {
        return fromTable;
    }

    public void setFromTable(String fromTable) {
        this.fromTable = fromTable;
    }

    public String getFromAlias() {
        return fromAlias;
    }

    public void setFromAlias(String fromAlias) {
        this.fromAlias = fromAlias;
    }

    public List<TableJoin> getJoins() {
        return joins;
    }

    public List<String> getRestrictions() {
        return restrictions;
    }

    public void setRestrictions(List<String> restrictions) {
        this.restrictions = restrictions;
    }

    public List<SQLField> getFields() {
        return Collections.unmodifiableList(fields);
    }
}
